`timescale 1ns/100fs

module testbench ;

// General parameters for the testbench
parameter real supply_voltage=1.1 ;
parameter real tr=30.0e-12;

// Input events are generated every tic_delay
localparam tic_delay= 1;

// Testbench signals
bit clk, d ; // Generated signals may only have 0/1 values
wire q ;     // Output signals are simple wires.
real vddval=supply_voltage ; // The testbench defines the value of the power supply
                             // allowing this value to be changed during simulation.

// The board itself. 
board  bdut( .d(d), .clk(clk), .q(q),.vddval(vddval)) ;


//// As we want to check the behavior of our D flip-flop, we may use a fully
//// digital reference model of a D flip-flop and compare the output results 
//// of the DUT with those of the reference model.
logic qref ;
always @(posedge clk)
    qref <= d ;

// A sequence of (d,clk) pairs for exhaustive test
// all transitions of d are tested while clk is 0
// all transitions of d are tested while clk is 1
// all transitions of clk are tested while d is 0
// all transitions of clk are tested while d is 1
// Values are stored in a array.
bit [1:0] valseq [] = {2'b00, 2'b01, 2'b11, 2'b10, 2'b00, 2'b10, 2'b11, 2'b01, 2'b00} ;

// Defines a file name for storing output results
string outfilename ;
// Define a file pointer for the file.
int outfile ;
// Define the standard error channel.
integer STDERR = 32'h8000_0002;

// The testbench itself
initial
begin:tb
   // Forge the filename. Results are stored in the result directory
   // The name of the file is qualified by the choosen power supply value
   $swrite(outfilename,"results/test_dut_%3.1fV.dat",supply_voltage) ;
   // Open the file for writing
   outfile = $fopen(outfilename,"w") ;
   // Defines format for for writing timing infos 
   $timeformat(-9,2,"ns",20);
   // Loop on all values
   for(int i=0;i<valseq.size;i++) begin
       {d,clk} = valseq[i] ; // new values are generated for the inputs
       #tic_delay ;  // just wait tic_delay in order to have stable outputs
       if(i>0) begin // now we test q output consistancy. The first test is skipped as the electrical model may be in an uknown state
           // Store measured infos in the file
           $fdisplay(outfile,"testbench: at time %0t: d:%0d, clk:%0d, q:%0d, qref:%0d", $realtime,d,clk,q,qref) ;
           if(q==qref) // Display measured info on the console (green for good values)
              $fdisplay(STDERR,"\033[92m testbench: at time %0t: d:%0d, clk:%0d, q:%0d, qref:%0d \033[0m", $realtime,d,clk,q,qref) ;
           else
            begin // Display a red message, and stop simulation when an error is detected
              $fdisplay(STDERR,"\033[91m testbench: at time %0t: d:%0d, clk:%0d, q:%0d, qref:%0d \033[0m", $realtime,d,clk,q,qref) ;
              $fdisplay(STDERR,"\033[92m testbench: ERROR detected from testbench\033[0m" ) ;
              $fflush(STDERR) ;
              $fflush(outfile) ;
              $finish ;
            end
           end
       end
   #tic_delay;
   $fdisplay(STDERR,"\033[92m testbench: Result from analog simulation seems  OK\033[0m" ) ;
   $fclose(outfile);
   $stop;
end 

endmodule
