function mat_cpa=cpa(sbox_n,tab_Ind,ciph_dec)
  
  % Last round of AES:
  % 
  %  X_1   ...  ...  ...   X_16 
  %   |                     |
  % +---+                 +---+
  % | S |  ...  ...  ...  | S |
  % +---+                 +---+
  %   |     |    |    |     |
  % +-------------------------+
  % |        Shift Rows       |
  % +-------------------------+
  %   |     |    |    |     |
  %  +-+                   +-+
  %  |+|--R_1   ...  ...   |+|--R_16
  %  +-+                   +-+
  %   |                     |
  %  Y_1   ...  ...  ...   Y_16
  %
  % X_i: State before last round
  % Y_i: Cipher text
  %
  % To guess subkey #i (which is R_i in the schema), we compute the
  % hamming weight of Y_j xor X_j, where j is the inverse shift row
  % function applied to i. To compute X_j, we will go backwards from
  % Y_i all the way up the schema:
  % - Apply hypothetical round key
  % - Do the inverse shift row
  % - Apply the corresponding inverse S-Box

  % Pre-compute lookup-table of all possible hamming weights
  hw_tab=[];
  for w=1:256
    hw_tab(w)=hw(w-1);
  end;
  
  % Table to find the index of the S-Box to apply
  invShiftRows_sbox=[1 6 11 16 5 10 15 4 9 14 3 8 13 2 7 12];

  % Cipher text bytes
  Y_i = ciph_dec(:,sbox_n); 
  Y_j = ciph_dec(:,invShiftRows_sbox(sbox_n));

  % Vector of key hypotheses
  K=0:255;

  % Apply hypothetical keys 
  tmp = bitxor( Y_i*ones(1,length(K)) , ones(length(Y_i),1)*K );

  % Apply inverse S-Box
  X_j = InvSbox(tmp);

  % Compute hamming distance
  dst = bitxor( Y_j*ones(1,length(K)), X_j );
  hd  = hw_tab( dst+1 );

  % Normalize prediction and observation
  n_measures = size(tab_Ind,1);
  Prediction  = (hd - 4) / std(hw_tab);
  Observation = (tab_Ind - ones(n_measures,1)*mean(tab_Ind)) ./ (ones(n_measures,1) * std(tab_Ind));  

  % Compute correlation
  mat_cpa = (Observation' * Prediction) / n_measures;
  
